using System;
using System.IO;

using Team_Project.PersistencyManagers.Storages;

namespace Team_Project.Elements
{
	/// <summary>
	/// Classe base per la creazione di elementi di team project
	/// </summary>
	public abstract class IProjectElement :MarshalByRefObject, IDisposable
	{
		/// <summary>
		/// Mantiene il nome del progetto di cui l'istanza fa parte
		/// </summary>
		protected string project;

		/// <summary>
		/// Ottiene il nome del file
		/// </summary>
		public abstract string Name
		{
			get;
		}

		/// <summary>
		/// Ottiene il nome del progetto di cui l'istanza fa parte
		/// </summary>
		public virtual string Project
		{
			get{return project;}
		}

		/// <summary>
		/// Ottiene il suffisso aggiunto al termine del nome dell'elemento per identificarne
		/// il tipo in storages misti. L'implementazione di default restituisce
		/// la stringa vuota.
		/// </summary>
		public virtual string Suffix
		{
			get{return "";}
		}

		/// <summary>
		/// Restituisce il nome completo dell'elemento, composto da location e nome
		/// separati da una "/".
		/// </summary>
		public virtual string FullName
		{
			get{return Location + "/" + Name;}
		}

		/// <summary>
		/// Ottiene o imposta la location della location nella quale  contenuto
		/// l'elemento
		/// </summary>
		public abstract string Location
		{
			get;
			set;
		}

		/// <summary>
		/// Permette di impostare lo stream di input dal quale leggere i dati.
		/// Impostando questa propriet i dati vengono letti dallo stream.
		/// Questa propriet  generalmente utilizzata dalla funzione LoadFrom
		/// </summary>
		public abstract Stream DataInput
		{
			set;
		}

		/// <summary>
		/// Inizializza l'istanza dell'elemento leggendo i dati dallo storage
		/// passato come parametro.
		/// </summary>
		/// <param name="src">Storage dal quale leggere i dati</param>
		public abstract void LoadFrom(IStorage src);
		/// <summary>
		/// Scrive i dati sullo storage. Tutti i dati devono essere gi valorizzati
		/// e le modifiche successive non si ripercuotono sullo storage fino ad un
		/// successivo salvataggio
		/// </summary>
		/// <param name="dest">Storage destinazione dove scrivere i dati</param>
		public abstract void WriteTo(IStorage dest);
		/// <summary>
		/// Distrugge un elemento dallo storage in parametro.
		/// I dati memorizzati nell'istanza potrebbero
		/// non essere pi validi una volta distrutto.
		/// </summary>
		/// <param name="src">Storage dal quale eliminare l'elemento.</param>
		public abstract void DestroyFrom(IStorage src);

		/// <summary>
		/// Scrive i dati dell'elemento su uno stream
		/// </summary>
		/// <param name="dest">Stream destinazione</param>
		public abstract void WriteTo(Stream dest);

		#region IDisposable Members
		/// <summary>
		/// Rilascia le risorse occupate dall'oggetto
		/// </summary>
		public abstract void Dispose();

		#endregion
	}
}
